import numpy as np
import h5py


def load_samples(h5_path: str) -> np.ndarray:
    """
    Load field configuration samples from an HDF5 file.

    Args:
      h5_path: Path to the HDF5 file containing dataset 'samples'.

    Returns:
      samples: ndarray of shape (n_samples, N)
    """
    with h5py.File(h5_path, 'r') as f:
        samples = f['samples'][:]
    return samples


def compute_two_point_correlator(samples: np.ndarray) -> np.ndarray:
    """
    Compute the equal-time two-point correlator C_{ij} = <q_i q_j> - <q_i><q_j>.

    Args:
      samples: ndarray of shape (n_samples, N)

    Returns:
      correlator: ndarray of shape (N, N)
    """
    mean = np.mean(samples, axis=0)
    centered = samples - mean
    correlator = (centered.T @ centered) / samples.shape[0]
    return correlator


def estimate_beta_function(correlator: np.ndarray, scale_indices: list) -> float:
    """
    Estimate an effective beta-function via scale-dependence of the correlator.
    Uses the diagonal entries C_{ii} at specified non-zero scale indices.

    Args:
      correlator: ndarray of shape (N, N)
      scale_indices: list of int indices corresponding to scales

    Returns:
      beta: float slope from linear fit of ln C vs. ln scale
    """
    # Filter out non-positive indices to avoid log(0)
    safe_indices = [i for i in scale_indices if i > 0]
    if not safe_indices:
        raise ValueError("No positive scale indices provided for beta estimation.")

    diag_vals = correlator.diagonal()[safe_indices]
    ln_scales = np.log(np.array(safe_indices, dtype=float))
    ln_vals   = np.log(diag_vals)

    beta, _ = np.polyfit(ln_scales, ln_vals, 1)
    return float(beta)


def anomaly_indicator(samples: np.ndarray) -> float:
    """
    Placeholder for computing an anomaly indicator (e.g., divergence of a current).
    Returns 0.0 by default until a concrete definition is provided.

    Args:
      samples: ndarray of shape (n_samples, N)

    Returns:
      float anomaly indicator
    """
    # Default anomaly indicator
    return 0.0
